<?php
/**
 * Log system activity
 * 
 * @param string $action_type The type of action (login, logout, create, update, delete)
 * @param string $description Detailed description of the action
 * @param int|null $user_id The ID of the user performing the action (null for system actions)
 * @param string|null $module The module where the action occurred (e.g., 'users', 'inventory', 'sales')
 * @return bool True if logging was successful, false otherwise
 */
function logActivity($action_type, $description, $user_id = null, $module = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO system_logs (user_id, action, details, module, created_at)
            VALUES (:user_id, :action, :details, :module, NOW())
        ");
        
        return $stmt->execute([
            'user_id' => $user_id,
            'action' => $action_type,
            'details' => $description,
            'module' => $module
        ]);
    } catch (PDOException $e) {
        // Log error to error log
        error_log("Failed to log activity: " . $e->getMessage());
        return false;
    }
}

/**
 * Get color class for action type badge
 * 
 * @param string $action_type The type of action
 * @return string The Bootstrap badge class
 */
function getActionBadgeClass($action_type) {
    return match($action_type) {
        'login' => 'success',
        'logout' => 'secondary',
        'create' => 'primary',
        'update' => 'info',
        'delete' => 'danger',
        'error' => 'danger',
        'warning' => 'warning',
        default => 'secondary'
    };
}

/**
 * Format timestamp for display
 * 
 * @param string $timestamp The timestamp to format
 * @return string Formatted date and time
 */
function formatTimestamp($timestamp) {
    return date('M d, Y H:i:s', strtotime($timestamp));
}

/**
 * Get user-friendly action description
 * 
 * @param string $action_type The type of action
 * @return string User-friendly action description
 */
function getActionDescription($action_type) {
    return match($action_type) {
        'login' => 'User Login',
        'logout' => 'User Logout',
        'create' => 'Created New Record',
        'update' => 'Updated Record',
        'delete' => 'Deleted Record',
        'error' => 'System Error',
        'warning' => 'System Warning',
        default => ucfirst($action_type)
    };
}

/**
 * Log system actions to the database
 * 
 * @param string $action The action being performed
 * @param string $description Description of the action
 * @return bool True if logging was successful, false otherwise
 */
function logSystemAction($action, $description) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO system_logs (
                user_id,
                action,
                details,
                created_at
            ) VALUES (?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $_SESSION['user_id'] ?? null,
            $action,
            $description
        ]);
        
        return true;
    } catch (PDOException $e) {
        // Log to error log if database logging fails
        error_log("Failed to log system action: " . $e->getMessage());
        return false;
    }
}

/**
 * Get and display the store logo
 * @param string $class CSS class for the image
 * @param string $style Inline CSS styles
 * @param string $defaultLogo Path to default logo if store logo doesn't exist
 * @return string HTML img tag with the logo
 */
function getStoreLogo($class = 'store-logo', $style = 'max-width: 150px; max-height: 80px;', $defaultLogo = 'assets/images/default-logo.png') {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info' LIMIT 1");
        $stmt->execute();
        $logo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($logo && $logo['setting_binary']) {
            // Determine mime type from filename
            $mimeType = 'image/jpeg'; // Default to JPEG
            if (strpos($logo['setting_value'], '.png') !== false) {
                $mimeType = 'image/png';
            } elseif (strpos($logo['setting_value'], '.gif') !== false) {
                $mimeType = 'image/gif';
            }
            
            return '<img src="data:' . $mimeType . ';base64,' . base64_encode($logo['setting_binary']) . '" 
                    alt="Store Logo" 
                    class="' . htmlspecialchars($class) . '" 
                    style="' . htmlspecialchars($style) . '">';
        } else {
            return '<img src="' . htmlspecialchars($defaultLogo) . '" 
                    alt="Store Logo" 
                    class="' . htmlspecialchars($class) . '" 
                    style="' . htmlspecialchars($style) . '">';
        }
    } catch (PDOException $e) {
        error_log("Error getting store logo: " . $e->getMessage());
        return '<img src="' . htmlspecialchars($defaultLogo) . '" 
                alt="Store Logo" 
                class="' . htmlspecialchars($class) . '" 
                style="' . htmlspecialchars($style) . '">';
    }
}

/**
 * Check if user is logged in
 * 
 * @return bool True if user is logged in, false otherwise
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Add serial numbers for a product
 * @param int $product_id The product ID
 * @param array $serial_numbers Array of serial numbers
 * @param string $source_type The source type (manual_adjustment or supplier_receipt)
 * @param int $source_id The ID of the source record
 * @return array Array of successfully added serial numbers
 */
function addProductSerialNumbers($product_id, $serial_numbers, $source_type, $source_id) {
    global $pdo;
    $added_serials = [];
    
    try {
        $pdo->beginTransaction();
        
        $stmt = $pdo->prepare("
            INSERT INTO product_serial_numbers 
            (product_id, serial_number, source_type, source_id) 
            VALUES (?, ?, ?, ?)
        ");
        
        foreach ($serial_numbers as $serial) {
            $serial = trim($serial);
            if (empty($serial)) continue;
            
            try {
                $stmt->execute([$product_id, $serial, $source_type, $source_id]);
                $added_serials[] = $serial;
            } catch (PDOException $e) {
                // Skip duplicate serial numbers
                if ($e->getCode() == 23000) continue;
                throw $e;
            }
        }
        
        $pdo->commit();
        return $added_serials;
    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        error_log("Error adding serial numbers: " . $e->getMessage());
        throw $e;
    }
}

/**
 * Get serial numbers for a product
 * @param int $product_id The product ID
 * @param string $status Optional status filter
 * @return array Array of serial numbers
 */
function getProductSerialNumbers($product_id, $status = null) {
    global $pdo;
    
    $sql = "SELECT serial_number, status, created_at, source_type 
            FROM product_serial_numbers 
            WHERE product_id = ?";
    $params = [$product_id];
    
    if ($status) {
        $sql .= " AND status = ?";
        $params[] = $status;
    }
    
    $sql .= " ORDER BY created_at DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Update serial number status
 * @param int $product_id The product ID
 * @param string $serial_number The serial number
 * @param string $status The new status
 * @param int|null $source_id Optional source ID (e.g., sale_id for sold items)
 * @return bool Success status
 */
function updateSerialNumberStatus($product_id, $serial_number, $status, $source_id = null) {
    global $pdo;
    
    try {
        $sql = "UPDATE product_serial_numbers SET status = ?";
        $params = [$status];
        
        if ($source_id !== null) {
            $sql .= ", source_id = ?";
            $params[] = $source_id;
        }
        
        $sql .= " WHERE product_id = ? AND serial_number = ?";
        $params[] = $product_id;
        $params[] = $serial_number;
        
        $stmt = $pdo->prepare($sql);
        return $stmt->execute($params);
    } catch (Exception $e) {
        error_log("Error updating serial number status: " . $e->getMessage());
        return false;
    }
}

/**
 * Check if serial numbers are available for a product
 * @param int $product_id The product ID
 * @param array $serial_numbers Array of serial numbers to check
 * @return array Array with 'available' and 'unavailable' serial numbers
 */
function checkSerialNumbersAvailability($product_id, $serial_numbers) {
    global $pdo;
    
    try {
        $placeholders = str_repeat('?,', count($serial_numbers) - 1) . '?';
        $sql = "SELECT serial_number, status 
                FROM product_serial_numbers 
                WHERE product_id = ? AND serial_number IN ($placeholders)";
        
        $params = array_merge([$product_id], $serial_numbers);
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        $existing_serials = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        $result = [
            'available' => [],
            'unavailable' => []
        ];
        
        foreach ($serial_numbers as $serial) {
            if (!isset($existing_serials[$serial])) {
                $result['unavailable'][] = $serial;
            } elseif ($existing_serials[$serial] !== 'available') {
                $result['unavailable'][] = $serial;
            } else {
                $result['available'][] = $serial;
            }
        }
        
        return $result;
    } catch (Exception $e) {
        error_log("Error checking serial numbers availability: " . $e->getMessage());
        throw $e;
    }
}

/**
 * Get available serial numbers for a product
 * @param int $product_id The product ID
 * @return array Array of available serial numbers
 */
function getAvailableSerialNumbers($product_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT serial_number 
            FROM product_serial_numbers 
            WHERE product_id = ? AND status = 'available'
            ORDER BY created_at DESC
        ");
        $stmt->execute([$product_id]);
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (Exception $e) {
        error_log("Error getting available serial numbers: " . $e->getMessage());
        return [];
    }
}

/**
 * Send WhatsApp message
 * @param string $phone The recipient phone number
 * @param string $message The message to send
 * @param PDO $pdo Database connection
 * @return array Array with 'success' boolean and optional 'error' message
 */
function send_whatsapp($phone, $message, $pdo) {
    try {
        // Get WhatsApp settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
        $stmt->execute();
        $whatsappSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        // Check if WhatsApp is enabled
        if (($whatsappSettings['enable_whatsapp'] ?? '0') !== '1') {
            return ['success' => false, 'error' => 'WhatsApp is not enabled'];
        }
        
        // Get sender number
        $senderNumber = $whatsappSettings['whatsapp_sender_number'] ?? '';
        if (empty($senderNumber)) {
            return ['success' => false, 'error' => 'WhatsApp sender number not configured'];
        }
        
        // Clean phone number (remove + and spaces)
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Prepare the API request data
        $apiData = [
            'sender_number' => $senderNumber,
            'recipient_number' => $phone,
            'message' => $message
        ];
        
        // Make API call to send WhatsApp message
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, '../api/send_whatsapp.php');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($apiData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response === false) {
            return ['success' => false, 'error' => 'Failed to connect to WhatsApp API'];
        }
        
        $result = json_decode($response, true);
        
        if ($httpCode !== 200 || !$result) {
            return ['success' => false, 'error' => 'WhatsApp API error: ' . ($result['message'] ?? 'Unknown error')];
        }
        
        if ($result['success']) {
            // Log successful WhatsApp send
            logSystemAction('whatsapp_sent', "WhatsApp message sent to $phone: " . substr($message, 0, 100) . '...');
            return ['success' => true];
        } else {
            return ['success' => false, 'error' => $result['message'] ?? 'WhatsApp send failed'];
        }
        
    } catch (Exception $e) {
        error_log("WhatsApp send error: " . $e->getMessage());
        return ['success' => false, 'error' => 'WhatsApp send error: ' . $e->getMessage()];
    }
}

if (!function_exists('getNextBatchNumber')) {
    /**
     * Generates the next sequential batch number based on the highest numeric value
     * found in the supplier_receipts table.
     *
     * @param PDO $pdo The PDO database connection object.
     * @return string The next batch number, formatted to 5 digits with leading zeros.
     */
    function getNextBatchNumber($pdo) {
        $stmt = $pdo->query("SELECT batch_number FROM supplier_receipts");
        $all_batch_numbers = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $max_number = 0;
        foreach ($all_batch_numbers as $batch_number) {
            // Use a regular expression to extract only the numeric part of the batch number.
            // This correctly handles prefixes like 'MM_' or any other characters.
            if (preg_match('/(\d+)/', $batch_number, $matches)) {
                $number = intval($matches[1]);
                if ($number > $max_number) {
                    $max_number = $number;
                }
            }
        }

        // Increment the highest number and format it.
        $next_number = $max_number + 1;
        return str_pad($next_number, 5, '0', STR_PAD_LEFT);
    }
}

if (!function_exists('getNextManualBatchNumber')) {
    /**
     * Generates the next sequential manual batch number in format mm-001, mm-002, etc.
     * based on existing manual batch numbers in the supplier_receipts table.
     *
     * @param PDO $pdo The PDO database connection object.
     * @return string The next manual batch number, formatted as mm-xxx.
     */
    function getNextManualBatchNumber($pdo) {
        $stmt = $pdo->query("SELECT batch_number FROM supplier_receipts WHERE batch_number LIKE 'mm-%'");
        $manual_batch_numbers = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $max_number = 0;
        foreach ($manual_batch_numbers as $batch_number) {
            // Extract the numeric part from mm-xxx format
            if (preg_match('/^mm-(\d+)$/i', $batch_number, $matches)) {
                $number = intval($matches[1]);
                if ($number > $max_number) {
                    $max_number = $number;
                }
            }
        }

        // Increment the highest number and format it as mm-xxx
        $next_number = $max_number + 1;
        return 'mm-' . str_pad($next_number, 3, '0', STR_PAD_LEFT);
    }
} 